# grdm_stordlの本体
#
# 改訂履歴：
# 2024-07-10 H.Hayashi : Poetry利用版から移植
# 2024-08-08 H.Hayashi : download後のファイルリスト作成を追加
# 2025-09-12 H.Hayashi : interactive_flagと--yesオプション追加

import argparse
import io
import os
import textwrap
import zipfile

import requests

#from grdm_storsync import settings  # <- Poetry利用版
#from . import settings  # <- エラー..
import settings

# 実行時引数を取得する関数の定義
def get_execargs():
    
    # helpメッセージのフォーマット変更(引数名と説明の間で改行しないように)
    class MyHelpFormatter(argparse.RawTextHelpFormatter, argparse.HelpFormatter):
        def __init__(self, prog, indent_increment=2, max_help_position=30, width=None):
            super().__init__(prog, indent_increment, max_help_position, width)
    
    # helpメッセージ中のdescriptionとepilogの内容定義
    descrp_text="""
        This command downloads all contents under a storage folder on a GakuNin RDM project and
        store them into a local folder.
        """
    epilog_text="""
        See README.md for details.
        """
    # ArgumentParser呼び出し
    parser = argparse.ArgumentParser(
        description=textwrap.indent(textwrap.dedent(descrp_text).rstrip(),' '),
        epilog=textwrap.indent(textwrap.dedent(epilog_text).rstrip(),' '),
        formatter_class=MyHelpFormatter
    )

    # 必須引数 (※なくてもエラーにならないようにnargs='?'とdefault値を設定)
    parser.add_argument('config_file', type=str, nargs='?', default=None, help="config file path")

    # オプション引数
    parser.add_argument('-mp', '--mount_point', type=str, nargs='?', const=None, default=None,
                        help='local folder where downloaded contents will be stored')
    
    # オプション引数
    parser.add_argument('-y', '--yes', action='store_true', 
                        help='assume "yes" as answer to all prompts and run non-interactively')

    # 引数取得(「-h/--help」が指定されたときはヘルプを表示して終了)
    args = parser.parse_args()

    return args

# メインプログラム
def download():
    
    # 実行時引数の取得
    try:
        myargs = get_execargs()
    except:  # 引数不正 or 「-h/--help」の場合
        return 100, False
    config_file = myargs.config_file
    mount_point = myargs.mount_point
    no_interact = myargs.yes
    
    # configファイルのチェック
    if config_file:  # 引数指定のconfigファイルがある場合
        if not os.path.isfile(config_file):  # ファイルが存在しない
            print(f'ERROR: Config-file "{config_file}" not found.')
            return 110, False
        else:
            pass
        if not os.access(config_file, os.R_OK):  # ファイルにアクセスできない
            print(f'ERROR: Config-file "{config_file}" not readable.')
            return 111, False
        else:
            pass
    else:
        pass
    
    # configファイルの読み込み 
    try:
        config = settings.get_config(config_file)
    except:
        if config_file:
            print(f'ERROR: Failed to read config-file "{config_file}".')
            return 112, False
        else:
            print(f'ERROR: Failed to get configuration parameters.')
            return 113, False
    
    # 本スクリプトの変数に代入
    if not mount_point: mount_point = config.grdm_mount_point #  実行時引数が優先
    access_token = config.grdm_access_token
    wbapi_baseurl = config.grdm_wbapi_baseurl
    project_id = config.grdm_project_id
    storage_provider = config.grdm_storage_provider
    target_folder = config.grdm_target_folder
    if not no_interact: #  -y/--yesオプション指定が無ければ環境変数GRDM_INTERACTIVE_FLAGから
        if config.grdm_interactive_flag:
            no_interact = False  # interactive mode
        else:
            no_interact = True  # non-interactive mode
    else:
        pass
    
    # 必須パラメタのチェック
    if not mount_point:
        print(f'ERROR: Config-parameter "GRDM_MOUNT_POINT" not given.')
        return 114, False
    else:
        pass
    if not access_token:
        print(f'ERROR: Config-parameter "GRDM_ACCESS_TOKEN" not given.')
        return 115, False
    else:
        pass
    if not wbapi_baseurl:
        print(f'ERROR: Config-parameter "GRDM_WBAPI_BASEURL" not given.')
        return 116, False
    else:
        pass
    if not project_id:
        print(f'ERROR: Config-parameter "GRDM_PROJECT_ID" not given.')
        return 117, False
    else:
        pass
    if not storage_provider:
        print(f'ERROR: Config-parameter "GRDM_STORAGE_PROVIDER" not given.')
        return 118, False
    else:
        pass
    
    # ダウンロードの可否をユーザに確認
    print('-'*110)
    print(f'Data files will be downloaded from GakuNin RDM storage.')
    print(f'  GakuNin RDM project ID       : {project_id}')
    print(f'  From (Remote storage folder) : /{storage_provider}{target_folder}')
    print(f'  To (Local folder)            : {mount_point}')
    print('-'*110)
    if not no_interact:
        while True:
            choice = input('Do you really want to proceed? [yes/no]').lower()
            if choice in ['n', 'no']:
                return 120, False
            elif choice in ['y', 'ye', 'yes']:
                break
    else:
        pass
        
    # ダウンロード先のフォルダ(マウントポイント)のチェック
    if not os.path.isdir(mount_point):  # フォルダが存在しない
        print(f'ERROR: Mount point "{mount_point}" not found.')
        return 130, False
    else:
        pass
    if len(os.listdir(mount_point)) != 0:  # フォルダが空でない
        print(f'ERROR: Mount point "{mount_point}" not empty.')
        return 131, False
    else:
        pass
    if not os.access(mount_point, os.W_OK) :  # フォルダに書き込み権限がない
        print(f'ERROR: Mount point "{mount_point}" not writable.')
        return 132, False
    else:
        pass

    # リクエストヘッダーを事前に定義
    headers = {
                'Authorization': 'Bearer {}'.format(access_token),
            }
    
    # ダウンロード元のフォルダのidを取得
    if target_folder != '/':  # ストレージのトップではない場合
        list_folders = target_folder.strip("/").split("/")
        folder_id = "/"  # 最初はストレージのトップ 
        for folder_name in list_folders:
            api_url = wbapi_baseurl.rstrip('/') + '/v1/resources/' + project_id + '/providers/' \
                      + storage_provider + folder_id + '?meta='
            response = requests.get(api_url, headers=headers)
            if response.status_code != requests.codes.ok:
                print(f'ERROR: Failed to get folder info. (HTTP status code = {response.status_code})')
                return 133, False
            else:
                pass
            folder_found = False
            for data in response.json()["data"]:
                if data["attributes"]["kind"] == 'folder':
                    if data["attributes"]["name"] == folder_name:
                        folder_id = data["attributes"]["path"]
                        #folder_path = data["attributes"]["materialized"]
                        folder_found = True
                        break
                    else:
                        continue
                else:
                    pass
            if not folder_found:
                print(f'ERROR: Target folder "{target_folder}" not found in remote.')
                return 134, False
            else:
                pass
    else:  # ストレージのトップの場合
        folder_id = "/"
    
    # zipをダウンロード＆展開
    api_url = wbapi_baseurl.rstrip('/') + '/v1/resources/' + project_id + '/providers/' \
              + storage_provider + folder_id + '?zip='
    with (
        requests.get(api_url, headers=headers) as response,
        io.BytesIO(response.content) as bytes_io,
        zipfile.ZipFile(bytes_io) as zip,
    ):
        if response.status_code != requests.codes.ok:
            print(f'ERROR: Failed to download data files. (HTTP status code = {response.status_code})')
            return 140, False
        else:
            try:
                zip.extractall(mount_point)
                print(f'Download contents stored under "{mount_point}".')
            except:
                print(f'ERROR: Failed to extract data files from downloaded ZIP archive.')
                return 141, False
    
    # ダウンロード後のファイルリスト作成
    abs_mount_point = os.path.realpath(mount_point)
    file_latestdl = os.path.join(abs_mount_point, '.ListAfterLatestDL.txt')
    try:
        with open(file_latestdl, 'w', encoding="utf-8") as f:
            for root, _, files in os.walk(abs_mount_point):
                for file in files:
                    f_path = os.path.join(root, file)
                    if ( f_path == file_latestdl ): continue
                    f_mtime = os.path.getmtime(f_path)
                    f.write(f'{f_path}|{f_mtime}' + "\n")
    except:
        print(f'ERROR: Failed to create a list of downloaded files.')
        return 150, False
    
    return 0, no_interact
