# Configファイルの読み込みプログラム
#
# 改訂履歴：
# 2024-07-10 H.Hayashi : Poetry利用版から移植(現状は完全コピー)
# 2025-08-08 H.Hayashi : grdm_upload_replace_flagをboolからintに変更
# 2025-09-12 H.Hayashi : grdm_interactive_flagを追加(default=True)

import json
from mimetypes import guess_type
import os
from typing import Optional

from dotenv import load_dotenv
from pydantic_settings import BaseSettings

class Settings(BaseSettings):
    
    # 以下は設定されない場合はNoneになり、処理がエラーになる
    # パーソナルアクセストークン
    grdm_access_token: Optional[str] = None
    # WaterButlet APIのベースURL (e.g. "https://dmsgrdm.riken.jp:7777/")
    grdm_wbapi_baseurl: Optional[str] = None
    # ダウンロード/アップロードするデータを所有するGRDMのプロジェクトのID (5桁のGUID)
    grdm_project_id: Optional[str] = None
    # ダウンロード/アップロードするデータを格納するストレージ名 (e.g. "s3compatriken")
    grdm_storage_provider: Optional[str] = None
    # データのダウンロード先/アップロード元になるローカルのフォルダのパス
    grdm_mount_point: Optional[str] = None
    
    # 以下は設定されない場合はdefault値が使われる
    # データのダウンロード元/アップロード先になるGRDMのストレージのフォルダのパス [defaultはストレージのトップ]
    grdm_target_folder: str = "/"
    # データのアップロード先にオリジナルのファイルがある場合に上書きするかどうかのフラグ [defaultは上書きしない]
    grdm_upload_replace_flag: int = 0
    # データをアップロード後にローカルのデータを削除するかどうかのフラグ [defaultは削除しない]
    grdm_local_delete_flag: bool = False
    # データをアップロードする際に同時処理する数 [defaultは10処理] 
    grdm_upload_concurrent: int = 10
    # 処理を実行する前に、ユーザのYes/Noを確認するかどうかのフラグ [defaultは確認する]
    grdm_interactive_flag: bool = True
    
    # 以下は現在のダウンローダー/アップローダーでは使わないが、マウンターで使うので設定しておく
    # OSF APIのベースURL (e.g.  "https://dmsgrdm.riken.jp:8000/v2/")
    grdm_osfapi_baseurl: Optional[str] = None
    # 実行時のログを出力するファイルのパス (e.g. "/tmp/grdm_mounter_output.log")
    grdm_out_log: Optional[str] = None

def get_config(config_file):
    
    ## .envファイルを読む関数
    def read_config_env(config_file):
        
        # configファイルは.envもしくは同等の環境変数定義の形式
        # configファイルは呼出し元のプログラムでチェック済み(またはNone)である想定
        if config_file:
            load_dotenv(config_file, override=True)
        else:
            pass
    
        return Settings()
    
    ## JSON形式の設定ファイルを読む関数
    def read_config_jsn(config_file):
        
        # configファイルはJSON形式
        # configファイルは呼出し元のプログラムでチェック済みである想定
        with open(config_file, "rb") as f:
            json_dict = json.load(f)

        return Settings(**json_dict)
    
    ## 環境変数設定ファイルを探す関数
    def find_dotenv() -> None:
        
        # ファイル名は「.env」とする
        dotenv = '.env'
        env_path     = os.getcwd() + '/' + dotenv  # カレントディレクトリにあるファイルが最優先
        env_path_alt = os.path.dirname(__file__) + '/' + dotenv  # カレントになければメインプログラムと同じフォルダにあるファイルを探す
        
        # ファイルが見つかればパスを返す
        if os.path.isfile(env_path):
            if os.access(env_path, os.R_OK):
                return env_path
            else:
                print(f'WARNING: "{env_path}" found, but not readable.')
        else:
            if env_path_alt != env_path:
                if os.path.isfile(env_path_alt):
                    if os.access(env_path_alt, os.R_OK):
                        return env_path_alt
                    else:
                        print(f'WARNING: "{env_path_alt}" found, but not readable.')
                else:
                    pass
            else:
                pass
    
        return
    
    ## JSON形式かどうかを判定する関数
    def exam_json(config_file) -> bool:
        
        # 現状ではファイル拡張子で判定
        if guess_type(config_file)[0] == 'application/json':
            return True
        else:
            return False
   
    # ユーザ設定の取得(※以下の優先順位)
    # 1) 引数指定のconfigファイルで設定されている値
    # 2) .envファイルで設定されている値
    # 3) OSの環境変数の設定値
    # 4) Settings()で設定されているdefault値
    if config_file:  # 引数指定のconfigファイルがある場合
        
        # configファイルの読み込み
        if exam_json(config_file):  # JSONの場合
            config = read_config_jsn(config_file)
        else:  # JSONでなければ.envと同等の環境変数設定ファイルと仮定
            config = read_config_env(config_file)
    
    else:
        
        # .envファイルの読み込み (※無い場合はOSの環境変数の値か、それも無ければdefault値)
        env_file = find_dotenv()  # .envファイルが無い場合はNone
        config = read_config_env(env_file)
    
    return config 
