# grdm_storlsの本体
#
# 改訂履歴：
# 2024-07-10 H.Hayashi : Poetry利用版から移植

import argparse
import os
import textwrap

import requests

#from grdm_storsync import settings  # <- Poetry利用版
import settings

# 実行時引数を取得する関数の定義
def get_execargs():
    
    # helpメッセージのフォーマット変更(引数名と説明の間で改行しないように)
    class MyHelpFormatter(argparse.RawTextHelpFormatter, argparse.HelpFormatter):
        def __init__(self, prog, indent_increment=2, max_help_position=30, width=None):
            super().__init__(prog, indent_increment, max_help_position, width)
    
    # helpメッセージ中のdescriptionとepilogの内容定義
    descrp_text="""
        This command shows storage provider name(s) of a GakuNin RDM project.
        """
    epilog_text="""
        See README.md for details.
        """
    # ArgumentParser呼び出し
    parser = argparse.ArgumentParser(
        description=textwrap.indent(textwrap.dedent(descrp_text).rstrip(),' '),
        epilog=textwrap.indent(textwrap.dedent(epilog_text).rstrip(),' '),
        formatter_class=MyHelpFormatter
    )

    # 必須引数 (※なくてもエラーにならないようにnargs='?'とdefault値を設定)
    parser.add_argument('config_file', type=str, nargs='?', default=None, help="config file path")
    
    # オプション引数
    parser.add_argument('-p', '--project_id', type=str, nargs='?', const=None, default=None,
                        help='5-digit ID of a GakuNin RDM project.')

    # 引数取得(「-h/--help」が指定されたときはヘルプを表示して終了)
    args = parser.parse_args()

    return args

# メインプログラム
def list():
    
    # 実行時引数の取得
    try:
        myargs = get_execargs()
    except:  # 引数不正 or 「-h/--help」の場合
        return 300
    config_file = myargs.config_file
    project_id  = myargs.project_id
    
    # configファイルのチェック
    if config_file:  # 引数指定のconfigファイルがある場合
        if not os.path.isfile(config_file):  # ファイルが存在しない
            print(f'ERROR: Config-file "{config_file}" not found.')
            return 310
        else:
            pass
        if not os.access(config_file, os.R_OK):  # ファイルにアクセスできない
            print(f'ERROR: Config-file "{config_file}" not readable.')
            return 311
        else:
            pass
    else:
        pass
    
    # configファイルの読み込み 
    try:
        config = settings.get_config(config_file)
    except:
        if config_file:
            print(f'ERROR: Failed to read config-file "{config_file}".')
            return 312
        else:
            print(f'ERROR: Failed to get configuration parameters.')
            return 313
    
    # 本スクリプトの変数に代入
    if not project_id: project_id = config.grdm_project_id #  実行時引数が優先
    access_token = config.grdm_access_token
    osfapi_baseurl = config.grdm_osfapi_baseurl
    
    # 必須パラメタのチェック
    if not project_id:
        print(f'ERROR: Config-parameter "GRDM_PROJECT_ID" not given.')
        return 314
    else:
        pass
    if not access_token:
        print(f'ERROR: Config-parameter "GRDM_ACCESS_TOKEN" not given.')
        return 315
    else:
        pass
    if not osfapi_baseurl:
        print(f'ERROR: Config-parameter "GRDM_OSFAPI_BASEURL" not given.')
        return 316
    else:
        pass
    
    # プロジェクトにマウントされたストレージプロバイダの情報取得
    headers = {
            'Authorization': 'Bearer {}'.format(access_token),
        }
    api_url = osfapi_baseurl.rstrip('/') + '/nodes/' + project_id + '/files/'
    response = requests.get(api_url, headers=headers)
    if response.status_code != requests.codes.ok:
        print(f'ERROR: Failed to get storage provider info. [HTTP status code = {response.status_code}]')
        return 320
    else:
        pass

    # ストレージプロバイダ名を抽出
    storprovs = []  # ストレージプロバイダ名
    wbapiurls = []  # WaterButler APIのURL
    page = 1
    while True:
        res_dict = response.json()
        data_num = len(res_dict["data"])
        link_nxt = res_dict["links"]["next"]
        link_ttl = res_dict["links"]["meta"]["total"]
        link_ppg = res_dict["links"]["meta"]["per_page"]
        for i in range(0,data_num):
            storprov = res_dict["data"][i]["attributes"]["provider"]
            wbapiurl = res_dict["data"][i]["links"]["upload"]
            storprovs.append(storprov)
            wbapiurls.append(wbapiurl)
        if link_nxt:
            page += 1
            response = requests.get(link_nxt, headers=headers)  # 次のページの取得
            if response.status_code != requests.codes.ok:
                print(f'ERROR: Failed to get storage provider info. (Page={page}) [HTTP status code = {response.status_code}]')
            return 321
        else:
            break
    
    # ストレージプロバイダのリストをユーザに通知
    print('-'*110)
    print(f'Storages mounted to GakuNin RDM project `{project_id}` :')
    for (name, url) in zip(storprovs, wbapiurls):
        print(f'  {name}')
        #print(f'  {name}  (WaterButler API = {url})')
    print(f'Set one of above storage providers to environment variable "GRDM_STORAGE_PROVIDER"')
    print('-'*110)
    
    return 0
