# grdm_storulの本体
#
# 改訂履歴：
# 2024-07-10 H.Hayashi : Poetry利用版から移植
# 2025-08-08 H.Hayashi : upload_replace_flagを整数に変更し、2(追加・変更のみアップロード)を導入
# 2025-09-12 H.Hayashi : interactive_flagと--yesオプション追加
# 2025-09-17 H.Hayashi : osfstorage向け対応

import argparse
import datetime
#import importlib  # <- 移植版ではコメントアウト
#import json  # <- 移植版ではコメントアウト
import os
#import sys  # <- 移植版ではコメントアウト
import textwrap

import requests

#from grdm_storsync import settings  # <- Poetry利用版
import settings
#from grdm_storsync.grdm_storul import wbclient  # <- これがあるとうまくいかない..(なぜ？)  # <- Poetry利用版
#import wbclient  # <- これがあるとやっぱりダメなので、移植版もコメントアウト

# 実行時引数を取得する関数
def get_execargs():
    
    # helpメッセージのフォーマット変更(引数名と説明の間で改行しないように)
    class MyHelpFormatter(argparse.RawTextHelpFormatter, argparse.HelpFormatter):
        def __init__(self, prog, indent_increment=2, max_help_position=30, width=None):
            super().__init__(prog, indent_increment, max_help_position, width)
    
    # helpメッセージ中のdescriptionとepilogの内容定義
    descrp_text="""
        This command uploads all contents under a local folder into a storage folder on a GakuNin 
        RDM project.
        """
    epilog_text="""
        See README.md for details.
        """
    # ArgumentParser呼び出し
    parser = argparse.ArgumentParser(
        description=textwrap.indent(textwrap.dedent(descrp_text).rstrip(),' '),
        epilog=textwrap.indent(textwrap.dedent(epilog_text).rstrip(),' '),
        formatter_class=MyHelpFormatter
    )

    # 必須引数 (※なくてもエラーにならないようにnargsとdefault値を設定)
    parser.add_argument('config_file', type=str, nargs='?', default=None, help="config file path")

    # オプション引数
    parser.add_argument('-mp', '--mount_point', type=str, nargs='?', const=None, default=None,
                        help='local folder from which contents will be uploaded')
    
    # オプション引数
    parser.add_argument('-y', '--yes', action='store_true', 
                        help='assume "yes" as answer to all prompts and run non-interactively')

    # 引数取得(「-h/--help」が指定されたときはヘルプを表示して終了)
    args = parser.parse_args()

    return args

# フォルダ内を削除する関数
def del_folder_cont(folder_path) -> None:
    
    # フォルダ以下のファイル、サブフォルダを最深層から順次削除
    for root, dirs, files in os.walk(folder_path, topdown=False):
        for name in files:
            file_path = os.path.join(root, name)
            os.remove(file_path)
            #print(f'Local file "{file_path}" deleted.')  # 個別の削除情報は出力しない
        for name in dirs:
            dir_path = os.path.join(root, name)
            os.rmdir(dir_path)
            #print(f'Local directory "{dir_path}" deleted.')  # 個別の削除情報は出力しない
    
    return

# メインプログラム
def upload():
    
    # 実行時引数の取得
    try:
        myargs = get_execargs()
    except:  # 引数不正 or 「-h/--help」の場合
        return 200, False
    config_file = myargs.config_file
    mount_point = myargs.mount_point
    no_interact = myargs.yes
    
    # configファイルのチェック
    if config_file:  # 引数指定のconfigファイルがある場合
        if not os.path.isfile(config_file):  # ファイルが存在しない
            print(f'ERROR: Config-file "{config_file}" not found.')
            return 210, False
        else:
            pass
        if not os.access(config_file, os.R_OK):  # ファイルにアクセスできない
            print(f'ERROR: Config-file "{config_file}" not readable.')
            return 211, False
        else:
            pass
    else:
        pass
    
    # ユーザ設定の取得
    try:
        config = settings.get_config(config_file)
    except:
        if config_file:
            print(f'ERROR: Failed to read config-file "{config_file}".')
            return 212, False
        else:
            print(f'ERROR: Failed to get configuration parameters.')
            return 213, False
    
    # 本スクリプトの変数に代入
    if not mount_point: mount_point = config.grdm_mount_point #  実行時引数が優先
    access_token = config.grdm_access_token
    wbapi_baseurl = config.grdm_wbapi_baseurl
    project_id = config.grdm_project_id
    storage_provider = config.grdm_storage_provider
    target_folder = config.grdm_target_folder
    upload_concurrent = config.grdm_upload_concurrent
    upload_replace_flag = config.grdm_upload_replace_flag
    local_delete_flag = config.grdm_local_delete_flag
    if not no_interact: #  -y/--yesオプション指定が無ければ環境変数GRDM_INTERACTIVE_FLAGから
        if config.grdm_interactive_flag:
            no_interact = False  # interactive mode
        else:
            no_interact = True  # non-interactive mode
    else:
        pass
    
    # 必須パラメタのチェック
    if not mount_point:
        print(f'ERROR: Config-parameter "GRDM_MOUNT_POINT" not given.')
        return 214, False
    else:
        pass
    if not access_token:
        print(f'ERROR: Config-parameter "GRDM_ACCESS_TOKEN" not given.')
        return 215, False
    else:
        pass
    if not wbapi_baseurl:
        print(f'ERROR: Config-parameter "GRDM_WBAPI_BASEURL" not given.')
        return 216, False
    else:
        pass
    if not project_id:
        print(f'ERROR: Config-parameter "GRDM_PROJECT_ID" not given.')
        return 217, False
    else:
        pass
    if not storage_provider:
        print(f'ERROR: Config-parameter "GRDM_STORAGE_PROVIDER" not given.')
        return 218, False
    else:
        pass
    
    # wbclient向けの調整
    wbapi_baseurl = wbapi_baseurl.rstrip('/')  # 右端の'/'は無い想定(※requestsのバージョンが低いと対応できない)
    
    # 上書きしない場合のアップロード先の変更
    if upload_replace_flag == 0:
        tz_jst = datetime.timezone(datetime.timedelta(hours=9))  # タイムゾーンをJSTに設定
        dt_now = datetime.datetime.now(tz_jst)
        datetime_str = dt_now.strftime("%Y%m%d_%H%M%S")
        target_folder = target_folder.rstrip('/') + '/' + datetime_str + '/'
    else:
        target_folder = target_folder.rstrip('/') + '/'  # 上書きする場合で、最後に'/'を忘れているとエラーになるので
    
    # アップロードの可否をユーザに確認
    if upload_replace_flag == 1:
        upload_warning = "All files in the remote will be replaced by local files. [GRDM_UPLOAD_REPLACE_FLAG=1]"
    elif upload_replace_flag == 2:
        upload_warning = "Only added and modified files will be uploaded. [GRDM_UPLOAD_REPLACE_FLAG=2]"
    else:  # must be 0
        upload_warning = "New remote folder will be created and local files will uploaded there. [GRDM_UPLOAD_REPLACE_FLAG=0]"
    if local_delete_flag:
        delete_warning = "Local files will be deleted after uploading. [GRDM_LOCAL_DELETE_FLAG=TRUE]"
    else:
        delete_warning = "Local files will be kept after uploading. [GRDM_LOCAL_DELETE_FLAG=FALSE]"  
    print('-'*110)
    print(f'Data files will be uploaded to GakuNin RDM storage.')
    print(f'  GakuNin RDM project ID     : {project_id}')
    print(f'  From (Local folder)        : {mount_point}')
    print(f'  To (Remote storage folder) : /{storage_provider}{target_folder}')
    print(f'  * Note: {upload_warning}')
    print(f'  * Note: {delete_warning}')
    print('-'*110)
    if not no_interact:
        while True:
            choice = input('Do you really want to proceed? [yes/no]').lower()
            if choice in ['n', 'no']:
                return 220, False
            elif choice in ['y', 'ye', 'yes']:
                break
    else:
        pass
    
    # アップロード元のフォルダ(マウントポイント)のチェック
    if not os.path.isdir(mount_point):  # フォルダが存在しない
        print(f'ERROR: Mount point "{mount_point}" not found.')
        return 230, False
    else:
        pass
    if not os.access(mount_point, os.R_OK) :  # フォルダに読み出し権限がない
        print(f'ERROR: Mount point "{mount_point}" not readable.')
        return 231, False
    else:
        pass
    if len(os.listdir(mount_point)) == 0:  # フォルダ以下に何もない
        print(f'ERROR: No uploading contents found under "{mount_point}".')
        return 232, False
    else:
        pass
    
     # リクエストヘッダーを事前に定義
    headers = {
                'Authorization': 'Bearer {}'.format(access_token),
            }

    # アップロード先の準備
    if upload_replace_flag == 0:  # 新規作成(上書きしない場合)
        # 最下層のフォルダ名を取得
        list_folders = target_folder.strip("/").split("/")
        folder_tbc = list_folders[-1]
        # 親フォルダのidを取得
        if len(list_folders) > 1:  # 親フォルダがストレージのトップではない場合
            folder_id = "/"  # ストレージのトップから探す
            for i in range(len(list_folders)-1):
                folder_name = list_folders[i]
                api_url = wbapi_baseurl.rstrip('/') + '/v1/resources/' + project_id + '/providers/' \
                            + storage_provider + folder_id + '?meta='
                response = requests.get(api_url, headers=headers)
                if response.status_code != requests.codes.ok:
                    print(f'ERROR: Failed to get folder info. (HTTP status code = {response.status_code})')
                    return 240, False
                else:
                    pass
                folder_found = False
                for data in response.json()["data"]:
                    if data["attributes"]["kind"] == 'folder':
                        if data["attributes"]["name"] == folder_name:
                            folder_id = data["attributes"]["path"]
                            #folder_path = data["attributes"]["materialized"]
                            folder_found = True
                            break
                        else:
                            continue
                    else:
                        pass
                if not folder_found:
                    print(f'ERROR: Parent of target folder "{target_folder}" not found in remote.')
                    return 241, False
                else:
                    pass
        else:  # 親フォルダがストレージのトップの場合
            folder_id = "/"
        # アップロード先のフォルダ作成
        api_url = wbapi_baseurl.rstrip('/') + '/v1/resources/' + project_id + '/providers/' \
            + storage_provider + folder_id + '?kind=folder&name=' + folder_tbc
        response = requests.put(api_url, headers=headers)
        if response.status_code != requests.codes.created:
            print(f'ERROR: Failed to create GRDM storage folder for uploading. [HTTP status code = {response.status_code}]')
            return 242, False
        else:
            remote_path = storage_provider + '/' + target_folder.strip('/') + '/'
            print(f'"{remote_path}": "newly created."')
    else:  # 存在確認(上書きする場合)
        # アップロード先のフォルダのidを取得
        if target_folder != '/':  # ストレージのトップではない場合
            list_folders = target_folder.strip("/").split("/")
            folder_id = "/"  # ストレージのトップから探す
            for folder_name in list_folders:
                api_url = wbapi_baseurl.rstrip('/') + '/v1/resources/' + project_id + '/providers/' \
                           + storage_provider + folder_id + '?meta='
                response = requests.get(api_url, headers=headers)
                if response.status_code != requests.codes.ok:
                    print(f'ERROR: Failed to get folder info. (HTTP status code = {response.status_code})')
                    return 243, False
                else:
                    pass
                folder_found = False
                for data in response.json()["data"]:
                    if data["attributes"]["kind"] == 'folder':
                        if data["attributes"]["name"] == folder_name:
                            folder_id = data["attributes"]["path"]
                            #folder_path = data["attributes"]["materialized"]
                            folder_found = True
                            break
                        else:
                            continue
                    else:
                        pass
                if not folder_found:
                    print(f'ERROR: Target folder "{target_folder}" not found in remote.')
                    return 244, False
                else:
                    pass
        else:
            folder_id = "/"
        # アップロード先のフォルダの確認
        api_url = wbapi_baseurl.rstrip('/') + '/v1/resources/' + project_id + '/providers/' \
                    + storage_provider + folder_id + '?meta='
        response = requests.get(api_url, headers=headers)
        if response.status_code != requests.codes.ok:
            print(f'ERROR: Failed to locate GRDM storage folder for uploading. [HTTP status code = {response.status_code}]')
            return 245, False
        else:
            #print(f'"{storage_provider}/{target_folder.strip('/')}/": "already exists."')  # 出力しない
            pass
    
    # ダウンロードリストの読み込み
    abs_mount_point = os.path.realpath(mount_point)
    file_latestdl = os.path.join(abs_mount_point, '.ListAfterLatestDL.txt')
    list_latestdl = list()  # 初期化
    if upload_replace_flag == 2:  # 追加・更新のみアップロードのケース
        try:
            with open(file_latestdl, 'r', encoding='utf-8') as f:
                list_latestdl = f.readlines()
        except Exception as e:
            print(f'ERROR: Failed to read the list file of latest download ({file_latestdl}). [exception = {e}]')
            return 250, False
    else:
        pass
        
    # wbclient用の環境変数の再設定
    os.environ["WBCLIENT_ENTRY_POINT"] = wbapi_baseurl
    os.environ["WBCLIENT_TOKEN"] = access_token
    os.environ["WBCLIENT_STORAGE_PROVIDER"] = storage_provider
    os.environ["WBCLIENT_CONCURRENT"] = str(upload_concurrent)  # 文字型でないとエラーになる
    #importlib.reload(sys.modules["grdm_storsync.grdm_storul"])  # OK  <- なくてもいいみたい  # <- Poetry利用版
    #importlib.reload(sys.modules["wbclient"])  # 必要ないようなので、移植版ではコメントアウト
    #from grdm_storsync.grdm_storul import wbclient  # <- wbclient/__init__.pyが必要  # <- Poetry利用版
    import wbclient  # <- wbclient/__init__.pyが必要
    
    # データファイルのアップロード：wbclientを呼ぶ
    exception = wbclient.exceptions
    command = wbclient.upload.UploadCommand()
    try:
        result = command.upload(project_id, mount_point, upload_replace_flag, list_latestdl, file_latestdl, target_folder)
        #print(json.dumps(result, ensure_ascii=False, indent=4))  # wbclientで逐次出力するように変更
        print(f'Contents under "{mount_point}" uploaded.')
    except exception.WBClientException as e:
        print(f'ERROR: Failed to upload data files. [wbclient exception = {e}]')
        return 260, False
    
    # データファイルの削除
    if local_delete_flag:
        try:
            del_folder_cont(mount_point)
            print(f'Contents under "{mount_point}" deleted.') 
        except Exception as e:
            print(f'ERROR: Failed to delete contents under "{mount_point}". [exception = {e}]')
            return 270, False
    else:
        pass    
    
    return 0, no_interact
