import json
import signal
import sys

from r2dms_uploader.main import R2dmsUploader  # poetry版(このファイルを指定してexe化する場合もfromはr2dms_uploaderから)

MSG = None  # 全体サマリの"message"の値
    
def sig_handler(signum, frame) -> None:
    
    global MSG
    
    # シグナル別のメッセージ定義
    if signum == signal.SIGINT:  # Ctrl-C
        sig_str = '"SIGINT" signal'
        MSG = "Aborted. (SIGINT)"
    elif signum == signal.SIGTERM:  # killコマンド
        sig_str = '"SIGTERM" signal'
        MSG = "Aborted. (SIGTERM)"
    else:  # これは起こらないはず
        sig_str = '"Some signal (num={signum})"'
        MSG = f"Aborted. (Signal #: {signum})"
    
    # 標準出力にメッセージ出力
    print(f'{sig_str} caught; Process aborted.')
    
    # 終了(終了の前にfinallyが実行される)
    status_code = 128 + signum  # 128 + シグナル番号
    sys.exit(status_code)

def main() -> None:
    
    global MSG
    
    # 全体サマリデータの初期化
    summary = R2dmsUploader().ini_summary()
    
    # シグナルトラップ
    signal.signal(signal.SIGTERM, sig_handler)
    signal.signal(signal.SIGINT,  sig_handler)
    
    # 処理実行
    try:
        ## debug
        #print(1/0)
        rc = R2dmsUploader().controller(summary)
        ## check
        #print(f'[check:__main__.main] rc = {rc}')
        #print(f'[check:__main__.main] summary = {summary}')
        #print(f'[check:__main__.main] summary["num_proc_failed"] = {summary["num_proc_failed"]}')
        #print(f'[check:__main__.main] summary["num_proc_skipped"] = {summary["num_proc_skipped"]}')
        if rc == 0:
            if summary["num_proc_failed"] != 0 or summary["num_proc_skipped"] != 0:
                MSG = "Completed. (with NG/SKIPPED process(es))"
            else:
                MSG = "Completed." 
        else:
            MSG = f"Aborted. (rc={rc})" 
            sys.exit(rc)
    except Exception as e:
        print(f'UNEXPECTED ERROR : {e}')
        MSG = f"Aborted. (exception: {e})"
        sys.exit(1)
    finally:
        # 全体サマリ出力の前にkillコマンドやCntl-Cで終了しないように
        signal.signal(signal.SIGTERM, signal.SIG_IGN)
        signal.signal(signal.SIGINT,  signal.SIG_IGN)
        # 全体サマリデータの最終化
        R2dmsUploader().fin_summary(summary, MSG)
        # 全体サマリデータの出力
        result = {"summary": summary}
        outputfile = result["summary"]["output_filepath"]
        if outputfile:
            print(f"See `{outputfile}` for details.")
            with open(outputfile, 'w') as f:
                json.dump(result, f, indent=2, ensure_ascii=False)
                print(file=f)  # 最終行に改行を入れる
        else:
            print(json.dumps(result, ensure_ascii=False))
        # 元に戻す
        signal.signal(signal.SIGTERM, signal.SIG_DFL)
        signal.signal(signal.SIGINT,  signal.SIG_DFL)
        #input("Press ENTER to exit")  # exe化しない場合はコメントアウト
  
    
if __name__ == "__main__":
    main()
