# r2dms_uploaderのメインプログラム
#
# 改訂履歴:
# 2024-01-xx H.Hayashi : 初版作成
# 2025-09-05 H.Hayashi : 一般化対応(理研用環境変数追加、等)
# 2025-09-22 H.Hayashi : 非Poetry版(importの変更)

import argparse
import datetime
import getpass
import json
import os
import socket
#import sys  # debug等で必要な時にimportする
#import time  # debug等で必要な時にimportする

import pandas  # CSVの読み取り
import requests  # GRDMプロジェクトURL取得

#from r2dms_uploader.settings import settings  # ※各ツールよりも先にimportする
from r2dms_uploader.settings import GRDM_TOKEN, DMSUTIL_TOKEN, GRDM_OSFAPI, GRDM_RIKEN  # ※各ツールよりも先にimportする
#sys.path.append('./r2dms_uploader')  # __init__.pyで実行するように変更
#from create_grdmprj.main import create_grdmprj  # GRDMプロジェクト作成ツール  # poetry版
#from r2dms_dmsutil.main import r2dms_dmsutil  # S3バケット作成ツール  # poetry版
from .create_grdmprj.main import create_grdmprj  # GRDMプロジェクト作成ツール  " exe化の際に「.」が必要
#from .r2dms_dmsutil.main import r2dms_dmsutil  # S3バケット作成ツール  # exe化の際に「.」が必要  # NII版ではコメントアウト
#import wbclient  # データアップロードツール ※環境変数の更新があるので、invoke_wbclient関数内でimportする

class R2dmsUploader:

    # サマリデータの初期化
    def ini_summary(self):
        
        # 現在時刻(開始時刻)の取得
        curtime = datetime.datetime.now().astimezone().isoformat()

        # ホスト名の取得
        wrkhost = socket.gethostname()
        
        # カレントディレクトリの取得
        wrkdir  = os.getcwd()
        
        # ユーザIDの取得
        userid  = getpass.getuser()
        
        # サマリデータの定義と初期化
        summary = {
            "message" : None,
            "start_date": curtime,
            "stop_date": None,
            "work_host": wrkhost,
            "work_dir": wrkdir,
            "exec_user_id": userid,
            "input_filepath": None,
            "output_filepath": None,
            "num_csvline_valid": 0,
            "num_proc_total": 0,
            "num_proc_succeeded": 0,
            "num_proc_failed": 0,
            "num_proc_skipped": 0,
            "processes": None
        }
        
        return summary
    
    # サマリデータ入力の完了
    def fin_summary(self, summary, message):

        # メッセージの格納
        summary["message"] = message
        
        # 現在時刻(終了時刻)の取得と格納
        curtime = datetime.datetime.now().astimezone().isoformat()
        summary["stop_date"] = curtime

        #return summary
        return
    
    # 実行時引数の取得
    def get_execargs(self):
    
        parser = argparse.ArgumentParser()
        
        # 必須引数
        parser.add_argument('input', type=str, help="input file path")
        
        # オプション引数(-o/--output "output file path")
        parser.add_argument('-o', '--output', type=str, help='output file path')
        
        # オプション引数(--riken) -> 廃止
        #parser.add_argument('--riken', action='store_true', help=argparse.SUPPRESS)
        
        # 引数取得(「-h/--help」が指定されたときはヘルプを表示して終了)
        args = parser.parse_args()
        
        ## check
        #print(f'[check:get_execargs] args = {args}')
        
        return args
    
    # 入出力ファイルのチェック
    def exam_inout(self, inputfile, outputfile):
        
        # 入力ファイル確認
        if not os.path.isfile(inputfile):  # ファイルが存在しない
            print(f'ERROR: "{inputfile}" not found.')
            return -1
        else:
            pass
        if not os.access(inputfile, os.R_OK):  # ファイルにアクセスできない
            print(f'ERROR: "{inputfile}" not readable.')
            return -2
        else:
            pass
        
        # 出力ファイル確認
        if outputfile:  # 出力ファイルが指定されている場合のみ
            if os.path.isfile(outputfile):  # ファイルが既に存在する
                print(f'ERROR: "{outputfile}" already exists.')
                return -3
            else:
                pass
            outputdir = os.path.dirname(outputfile)
            if not outputdir:
                outputdir = "."  # ファイル名のみの場合はカレントディレクトリを想定
            else:
                pass
            if not os.access(outputdir, os.W_OK):  # ディレクトリに書き込み権限がない
                print(f'ERROR: "{outputdir}" not writable.')
                return -4
            else:
                pass              
        else:
            pass
        
        return 0
    
    # 入力ファイルの読み取り
    def read_inputs(self, filepath):

        # CSVの読み取り
        try:
            df = pandas.read_csv(filepath, encoding="utf-8", comment='#', header=0, skipinitialspace=True, dtype=str)
        except pandas.errors.EmptyDataError as e:  # 空ファイル
            print(f'ERROR: Failed to read {filepath}. ({e})')
            return -11, None
        except pandas.errors.ParserError as e:  # 要素数が過多のデータ行がある(※1行目以外)
            print(f'ERROR: Failed to read {filepath}. ({e})')
            return -12, None
        except Exception as e:
            print(f'ERROR: Failed to read {filepath}. ({e})')
            return -13, None
        ## check
        #print(f'[check:read_inputs] df = {df}')
        #print(f'[check:read_inputs] df.shape = {df.shape}')
        
        # 要素数定義
        if GRDM_RIKEN: # 理研向け
            num_elems_max = 11 # storage_providerとupload_folder指定時
            num_elems_min = 9  # storage_providerとupload_folder未指定時
        else:  # 一般向け
            num_elems_max = 6
            num_elems_min = 4
            
        # 想定外のフォーマットの場合はNG
        if df.shape[0] < 1:  # ヘッダーのみでデータ行がない
            print(f'ERROR: Valid CSV data line(s) not found in {filepath}.')
            return -14, None
        else:
            pass
        if df.shape[1] < num_elems_min:  # ヘッダーの要素数が不足
            print(f'ERROR: Number of CSV elements in {filepath} not enough (<{num_elems_min}).')
            return -15, None
        else:
            pass
        if df.shape[1] > num_elems_max:  # ヘッダーの要素数が過多
            print(f'ERROR: Number of CSV elements in {filepath} too many (>{num_elems_max}).')
            return -16, None
        else:
            pass
        # TODO : 最初のデータ行の要素数が過多でもexceptに引っ掛からないが現状ではトラップできない
        # TODO : - df.shape[1]はヘッダーの列数になってしまう..
        # TODO : - df.valuesはヘッダーの列数に合うようにデータが改変される..
        
        # CSVの空白がnanではなくNoneになるように変換
        df = df.where(df.notnull(), None)
        
        # データフレームからリストに変換
        csvlist = df.values.tolist()
        ## check
        #print(f'[check:read_inputs] csvlist = {csvlist}')
        
        return 0, csvlist
    
    # 環境変数のチェック
    def exam_environ(self):
        
        # 環境変数取得
        grdm_tkn = GRDM_TOKEN
        grdm_osf = GRDM_OSFAPI
        dmsu_tkn = DMSUTIL_TOKEN        
        ## check
        #print(f'[check:exam_environ] grdm_tkn = {grdm_tkn}')
        #print(f'[check:exam_environ] grdm_osf = {grdm_osf}')
        #print(f'[check:exam_environ] dmsu_tkn = {dmsu_tkn}')
        
        # 環境変数の未定義確認
        if GRDM_RIKEN:  # 理研向け
            if not grdm_tkn and not dmsu_tkn:  # 両トークンがいずれも未定義はあり得ない
                print('ERROR: Environment variables "GRDM_TOKEN" and "DMSUTIL_TOKEN" both undefined.')
                return -21
            else:
                pass
        else:  # 一般向け
            if not grdm_tkn:  # GRDMトークンは必須
                print('ERROR: Environment variable "GRDM_TOKEN" undefined.')
                return -22
            else:
                pass
            #if not grdm_osf:  # OSF APIは必須
            #    print('ERROR: Environment variable "GRDM_OSFAPI" undefined.')
            #    return -23
            #else:
            #    pass
        
        return 0
        
    # プロセスごとのサマリの初期化
    def ini_summary_proc(self, proc_id, csv_input):
        
        # 現在時刻(開始時刻)の取得
        curtime = datetime.datetime.now().astimezone().isoformat()

        # プロセスIDの桁調整
        procid5 = str(proc_id).zfill(5)
        
        # サマリデータの定義と初期化
        if GRDM_RIKEN:  # 理研向け
            summary = {
                "proc_id": procid5,
                "status": None,
                "message" : "In progress.",
                "start_date": curtime,
                "stop_date": None,
                "csv_input": csv_input,
                "create_grdmprj": None,
                "r2dms_dmsutil": None,
                "wbclient": None,
            }
        else:  # 一般向け -> "r2dms_dmsutil"を除外
            summary = {
                "proc_id": procid5,
                "status": None,
                "message" : "In progress.",
                "start_date": curtime,
                "stop_date": None,
                "csv_input": csv_input,
                "create_grdmprj": None,
                "wbclient": None,
            }
        
        return summary        
    
    # プロセスごとのサマリ入力の完了               
    def fin_summary_proc(self, summary, status, message):

        # ステータスの格納
        summary["status"] = status
        
        # メッセージの格納
        summary["message"] = message
        
        # 現在時刻(終了時刻)の取得と格納
        curtime = datetime.datetime.now().astimezone().isoformat()
        summary["stop_date"] = curtime

        return
     
    # ツール「create_grdmprj」の呼びだし
    def invoke_create_grdmprj(self, sum_proc):
        
        # <ローカル関数定義開始>
        
        # 関数1：サマリの初期化
        def ini_summary_subproc():
            
            # 現在時刻(開始時刻)の取得と格納
            curtime = datetime.datetime.now().astimezone().isoformat()
        
            # サマリデータの定義と初期化
            summary_subproc = {
                "status": None,
                "start_date": curtime,
                "stop_date" : None,
                "message" : "In progress.",
                "internal_rc": None,
                "parent_guid": None,
                "root_guid": None,  # 2025-09-05追加(create_grdmprjのアップデートに合わせて)
                "guid_created": None,
                "url_created": None,
                "title_created": None,
                "creator_guid": None,
                "creator_name": None    
            }
            
            return summary_subproc
        
        # 関数2：サマリ入力の完了
        def fin_summary_subproc(summary_subproc, status):
      
            # ステータスの格納
            summary_subproc["status"] = status
            
            # 現在時刻(終了時刻)の取得と格納
            curtime = datetime.datetime.now().astimezone().isoformat()
            summary_subproc["stop_date"] = curtime
            
            return
        
        # 関数3：環境変数のチェック
        def exam_environ_subproc(summary_subproc):
            
            # CREATE_GRDMPRJ_TOKEN
            envvar = os.environ.get("CREATE_GRDMPRJ_TOKEN")
            if not envvar:
                summary_subproc["message"] = 'Environment variable "CREATE_GRDMPRJ_TOKEN" not defined.'
                return -1
            else:
                pass
            # CREATE_GRDMPRJ_OSFAPI
            envvar = os.environ.get("CREATE_GRDMPRJ_OSFAPI")
            if not envvar:
                summary_subproc["message"] = 'Environment variable "CREATE_GRDMPRJ_OSFAPI" not defined.'
                return -2
            else:
                pass
            
            return 0
        
        # <ローカル関数定義終了>
        
        # サブプロセスのサマリデータの初期化
        sum_subproc = ini_summary_subproc()
        
        # 親プロセスのサマリにサブプロセスのサマリを追加
        sum_proc["create_grdmprj"] = sum_subproc
        
        # 入力データ取得
        parent_id = sum_proc["csv_input"][0]
        prj_title = sum_proc["csv_input"][1]
        ## check
        #print(f'[check:invoke_create_grdmprj] parent_id = {parent_id}')
        #print(f'[check:invoke_create_grdmprj] prj_title = {prj_title}')
        
        # プロジェクトタイトルがNoneの場合は、スキップ
        if not prj_title:
            mystat = "SKIPPED"
            sum_proc["create_grdmprj"]["message"] = "Project title not given; Existing project will be used."
            sum_proc["create_grdmprj"]["parent_guid"] = parent_id  # 親プロジェクトIDは与えられているはず
            fin_summary_subproc(sum_proc["create_grdmprj"], mystat)
            return
        else:
            pass            
        
        # 環境変数の確認
        rc = exam_environ_subproc(sum_proc["create_grdmprj"])
        if rc != 0:
            mystat = "NG"
            fin_summary_subproc(sum_proc["create_grdmprj"], mystat)
            return
        else:
            pass
        
        # create_grdmprjの呼び出し
        sum_cg = create_grdmprj(parent_id, prj_title)
        ## check
        #print(f'[check:invoke_create_grdmprj] sum_cg= {sum_cg}')
        
        # create_grdmprjからの出力を受け取り、親プロセスのサマリを更新
        sum_proc["create_grdmprj"]["message"]       = sum_cg["message"]
        sum_proc["create_grdmprj"]["internal_rc"]   = sum_cg["internal_rc"]
        sum_proc["create_grdmprj"]["parent_guid"]   = sum_cg["parent_guid"]
        sum_proc["create_grdmprj"]["root_guid"]     = sum_cg["root_guid"]
        sum_proc["create_grdmprj"]["guid_created"]  = sum_cg["guid_created"]
        sum_proc["create_grdmprj"]["url_created"]   = sum_cg["url_created"]
        sum_proc["create_grdmprj"]["title_created"] = sum_cg["title_created"]
        sum_proc["create_grdmprj"]["creator_guid"]  = sum_cg["creator_guid"]
        sum_proc["create_grdmprj"]["creator_name"]  = sum_cg["creator_name"]
        
        # 内部リターンコードが0でない時はNG
        if sum_proc["create_grdmprj"]["internal_rc"] != 0:
            mystat = "NG"
        else:
            mystat = "OK"
        
        # 親プロセスのサマリを更新して呼び出し元に戻す
        fin_summary_subproc(sum_proc["create_grdmprj"], mystat)
        return
    
    # ツール「r2dms_dmsutil」の呼びだし
    def invoke_r2dms_dmsutil(self, sum_proc):
    
        # <ローカル関数定義開始>
        
        # 関数1：サマリの初期化
        def ini_summary_subproc():
            
            # 現在時刻(開始時刻)の取得と格納
            curtime = datetime.datetime.now().astimezone().isoformat()
        
            # サマリデータの定義と初期化
            summary_subproc = {
                "status": None,
                "start_date": curtime,
                "stop_date" : None,
                "message" : "In progress",
                "applicants": None,
                "lab_name_j": None,
                "name_j": None,
                "riken_s3bucket_name": None,
                "resfield_url": None,
                "total_data_size": None,
                "total_data_number": None,
                "remarks": None,
            }
            
            return summary_subproc
        
        # 関数2：サマリ入力の完了
        def fin_summary_subproc(summary_subproc, status=None, message=None):
      
            # ステータスの格納
            if status:
                summary_subproc["status"] = status
            else:
                pass
            
            # メッセージの格納
            if message:
                summary_subproc["message"] = message
            else:
                pass
            
            # 現在時刻(終了時刻)の取得と格納
            curtime = datetime.datetime.now().astimezone().isoformat()
            summary_subproc["stop_date"] = curtime
            
            return  
        
        # 関数3：環境変数のチェック
        def exam_environ_subproc(summary_subproc):
            
            # R2DMS_DMSUTIL_TOKEN
            envvar_name = "R2DMS_DMSUTIL_TOKEN"
            envvar = os.environ.get(envvar_name)
            if not envvar:
                summary_subproc["status"]  = "NG"
                summary_subproc["message"] = f"Environment variable '{envvar_name}' not defined."
                return -1
            else:
                pass
            
            return 0
        
        # 関数4：GRDMプロジェクトURL取得
        def get_grdmprj_url(project_guid):
            
            # 環境変数取得
            grdm_tkn = os.environ.get("CREATE_GRDMPRJ_TOKEN")
            grdm_api = os.environ.get("CREATE_GRDMPRJ_OSFAPI")
            if grdm_api is None or grdm_api is None:
                return_code = -1
                return return_code, None
            else:
                pass
            
            # ノード情報取得
            headers = {
                'Authorization': 'Bearer {}'.format(grdm_tkn),
            }
            api_url = grdm_api.rstrip('/') + '/nodes/' + project_guid + '/'
            response = requests.get(api_url, headers=headers)
            if response.status_code != requests.codes.ok:
                return_code = -1
                return return_code, None
            else:
                pass
            
            # プロジェクトURL取得
            res_lst = response.json()
            prj_url = res_lst["data"]["links"]["html"]
            return_code = 0
            
            return return_code, prj_url
        
        # 関数5：S3バケットのチェック
        def exam_s3bucket(indata_json, summary_subproc):
            
            # S3バケットのアドオン確認
            headers = {
                'Content-Type': 'application/json',
            }
            api_url = 'https://dmsutil.riken.jp/r2dms_dmsutil/r2dmsutil_dmsutil_addon_check.php/'  # r2dms_dmsutilと同様にハードコードしているが、要検討
            response = requests.post(api_url, headers=headers, data=indata_json.encode())
            if response.status_code != requests.codes.ok:
                return_code = -1
                summary_subproc["message"] = f"S3bucket check error (status_code={response.status_code})"
                return return_code
            else:
                res_lst = response.json()
                ## check
                #print(f'[check:invoke_r2dms_dmsutil/exam_s3bucket] response = {response}')
                #print(f'[check:invoke_r2dms_dmsutil/exam_s3bucket] res_lst = {res_lst}')
            
            # 確認結果の取得
            if res_lst["summary"]["status"] == "OK":
                return_code = 0
            else:
                return_code = 1
            summary_subproc["message"] = res_lst["summary"]["message"]
            if not res_lst["summary"]["bucket"]:
                summary_subproc["riken_s3bucket_name"] = None
            else:
                summary_subproc["riken_s3bucket_name"] = res_lst["summary"]["bucket"]
            
            return return_code
        
        # <ローカル関数定義終了>
        
        # サブプロセスのサマリデータの初期化
        sum_subproc = ini_summary_subproc()
        
        # 親プロセスのサマリにサブプロセスのサマリを追加
        sum_proc["r2dms_dmsutil"] = sum_subproc
        
        # 環境変数の確認
        rc = exam_environ_subproc(sum_proc["r2dms_dmsutil"])
        if rc != 0:  # 0以外はNGで終了
            fin_summary_subproc(sum_proc["r2dms_dmsutil"])
            return
        else:
            pass
        
        # 入力データ作成
        input_lst = {"indata": {
                "parent_id": sum_proc["create_grdmprj"]["parent_guid"],
                "project_id": sum_proc["create_grdmprj"]["guid_created"],
                "resfield_url": sum_proc["create_grdmprj"]["url_created"],
                "lab_cd": sum_proc["csv_input"][2],
                "id": sum_proc["csv_input"][3],
                "total_data_size": sum_proc["csv_input"][4],
                "total_data_number": sum_proc["csv_input"][5],
                "remarks": sum_proc["csv_input"][6],
                "data_path":sum_proc["csv_input"][7],
                "replace_flag":sum_proc["csv_input"][8]
            }
        }
        
        # 新規にプロジェクトが作成されたケースで、組織IDもしくは代表者IDがNoneの場合はスキップ
        if input_lst["indata"]["project_id"]:  # プロジェクトが作成されればproject_idは必ず存在する
            if not input_lst["indata"]["lab_cd"] or not input_lst["indata"]["id"]:
                mystatus  = "SKIPPED"
                mymessage = f"Valid `lab_cd` or `id` not given."
                fin_summary_subproc(sum_proc["r2dms_dmsutil"], mystatus, mymessage)
                return
            else:
                pass
        else:
            pass
        
        # 既存プロジェクトを使う場合はS3バケットのチェックを行う
        # - プロジェクトにアドオンされているS3バケットがある -> 既存としてスキップ
        # - プロジェクトにアドオンされているS3バケットがない -> プロジェクトURLを取得してS3バケット作成に進む
        if not input_lst["indata"]["project_id"]:  # create_grdmprjをスキップしているので、project_idはNoneのはず
            input_lst["indata"]["project_id"] = input_lst["indata"]["parent_id"]  # 既存プロジェクトIDをプロジェクトIDに代入
            input_jsn = json.dumps(input_lst, ensure_ascii=False)
            rc = exam_s3bucket(input_jsn, sum_proc["r2dms_dmsutil"])
            if rc == -1:  # APIエラーはNG
                mystatus  = "NG"
                fin_summary_subproc(sum_proc["r2dms_dmsutil"], mystatus)
                return
            else:
                if rc == 0:  # プロジェクトにアドオンされているS3バケットがある場合は既存としてスキップ
                    mystatus  = "EXISTING(CHECK-ONLY)"
                    fin_summary_subproc(sum_proc["r2dms_dmsutil"], mystatus)
                    return
                else:  # プロジェクトにアドオンされているS3バケットがない場合はプロジェクトURLを取得
                    prj_guid = input_lst["indata"]["project_id"]
                    rc, prj_url = get_grdmprj_url(prj_guid)
                    if rc != 0:  # プロジェクトURLが取得できない場合はNG
                        mystatus  = "NG"
                        mymessage = f"GakuNin RDM project (id={prj_guid}) not found."
                        fin_summary_subproc(sum_proc["r2dms_dmsutil"], mystatus, mymessage)
                        return
                    else:  # プロジェクトURLが取得できる場合はS3バケット作成に進む
                        input_lst["indata"]["resfield_url"] = prj_url
        else:
            pass        
        
        # r2dms_dmsutilの呼び出し
        ## check
        #print(f'[check:invoke_r2dms_dmsutil] input_lst = {input_lst}')
        #print(f'[check:invoke_r2dms_dmsutil] input_jsn = {input_jsn}')
        input_jsn = json.dumps(input_lst, ensure_ascii=False)
        sum_rd_jsn = r2dms_dmsutil(input_jsn)
        sum_rd_lst = sum_rd_jsn.json()
        ## check
        #print(f'[check:invoke_r2dms_dmsutil] sum_rd_jsn= {sum_rd_jsn}')
        #print(f'[check:invoke_r2dms_dmsutil] sum_rd_lst= {sum_rd_lst}')
        
        # r2dms_dmsutilからの出力を受け取り、親プロセスのサマリを更新
        sum_proc["r2dms_dmsutil"]["applicants"]          = sum_rd_lst["summary"]["output"]["applicants"]
        sum_proc["r2dms_dmsutil"]["lab_name_j"]          = sum_rd_lst["summary"]["output"]["lab_name_j"]
        sum_proc["r2dms_dmsutil"]["name_j"]              = sum_rd_lst["summary"]["output"]["name_j"]
        sum_proc["r2dms_dmsutil"]["riken_s3bucket_name"] = sum_rd_lst["summary"]["output"]["riken_s3bucket_name"]
        sum_proc["r2dms_dmsutil"]["resfield_url"]        = sum_rd_lst["summary"]["output"]["resfield_url"]
        sum_proc["r2dms_dmsutil"]["total_data_size"]     = sum_rd_lst["summary"]["output"]["total_data_size"]
        sum_proc["r2dms_dmsutil"]["total_data_number"]   = sum_rd_lst["summary"]["output"]["total_data_number"]
        sum_proc["r2dms_dmsutil"]["remarks"]             = sum_rd_lst["summary"]["output"]["remarks"]
    
        # ステータスとメッセージは、ツールの出力データに含まれるものをそのまま使う
        mystatus  = sum_rd_lst["summary"]["status"] 
        mymessage = sum_rd_lst["summary"]["message"]
        
        # 親プロセスのサマリを更新して呼び出し元に戻す
        fin_summary_subproc(sum_proc["r2dms_dmsutil"], mystatus, mymessage)
        return
    
    # ツール「wbclient」の呼びだし
    def invoke_wbclient(self, sum_proc):
        
        # <ローカル関数定義開始>
        
        # 関数1：サマリデータの初期化
        def ini_summary_subproc():
            
            # 現在時刻(開始時刻)の取得
            curtime = datetime.datetime.now().astimezone().isoformat()
        
            # サマリデータの定義と初期化
            summary_subproc = {
                "status": None,
                "start_date": curtime,
                "stop_date": None,
                "message": "In progress.",
                "local_path": None,
                "target_project_id": None,
                "storage_provider": None,
                "remote_dir": None,
                "replace_flag": None,
                "upload_result": None
            }
            
            return summary_subproc
        
        # 関数2：サマリデータ入力の完了
        def fin_summary_subproc(summary_subproc, status=None, message=None):
      
            # ステータスの格納
            if status:
                summary_subproc["status"] = status
            else:
                pass
            
            # メッセージの格納
            if message:
                summary_subproc["message"] = message
            else:
                pass
            
            # 現在時刻(終了時刻)の取得と格納
            curtime = datetime.datetime.now().astimezone().isoformat()
            summary_subproc["stop_date"] = curtime
            
            return
        
        # 関数3：環境変数のチェック
        def exam_environ_subproc(summary_subproc):
            
            # WBCLIENT_ENTRY_POINT
            envvar = os.environ.get("WBCLIENT_ENTRY_POINT")
            if not envvar:
                summary_subproc["message"] = 'Environment variable "WBCLIENT_ENTRY_POINT" not defined.'
                return -1
            else:
                pass
            # WBCLIENT_TOKEN
            envvar = os.environ.get("WBCLIENT_TOKEN")
            if not envvar:
                summary_subproc["message"] = 'Environment variable "WBCLIENT_TOKEN" not defined.'
                return -2
            else:
                pass
            # WBCLIENT_STORAGE_PROVIDER
            envvar = os.environ.get("WBCLIENT_STORAGE_PROVIDER")
            if not envvar:
                summary_subproc["message"] = 'Environment variable "WBCLIENT_STORAGE_PROVIDER" not defined.'
                return -3
            else:
                summary_subproc["storage_provider"] = envvar
            # WBCLIENT_REMOTE_PATH
            envvar = os.environ.get("WBCLIENT_REMOTE_PATH")
            if not envvar:
                summary_subproc["message"] = 'Environment variable "WBCLIENT_REMOTE_PATH" not defined.'
                return -4
            else:
                summary_subproc["remote_dir"] = envvar
            # WBCLIENT_CONCURRENT
            envvar = os.environ.get("WBCLIENT_CONCURRENT")
            if not envvar:
                summary_subproc["message"] = 'Environment variable "WBCLIENT_CONCURRENT" not defined.'
                return -5
            else:
                if not envvar.isdecimal():
                    summary_subproc["message"] = 'Environment variable "WBCLIENT_CONCURRENT" not integer.'
                    return -6
                else:
                    pass
                    
            return 0
        
        # <ローカル関数の定義終了>
        
        # サブプロセスのサマリデータの初期化
        sum_subproc = ini_summary_subproc()
        
        # 親プロセスのサマリにサブプロセスのサマリを追加
        sum_proc["wbclient"] = sum_subproc
        
        # 環境変数の確認
        rc = exam_environ_subproc(sum_proc["wbclient"])
        if rc != 0:
            mystat = "NG"
            fin_summary_subproc(sum_proc["wbclient"], mystat)
            return
        else:
            pass
        
        # アップロード用パラメタをCSVから取得
        if GRDM_RIKEN:  # 理研向け
            local_path = sum_proc["csv_input"][7]
            sum_proc["wbclient"]["local_path"] = local_path
            replace_flag = sum_proc["csv_input"][8]
            sum_proc["wbclient"]["replace_flag"] = replace_flag
            storage_provider = None
            if len(sum_proc["csv_input"]) > 9:
                storage_provider = sum_proc["csv_input"][9]
                sum_proc["wbclient"]["storage_provider"] = storage_provider
            else:  # 「storage_provider」がCSVで与えられていない場合は"s3compatriken"
                pass
            remote_dir = None
            if len(sum_proc["csv_input"]) > 10:
                remote_dir = sum_proc["csv_input"][10]
                sum_proc["wbclient"]["remote_dir"] = remote_dir
            else:  # 「remote_dir」がCSVで与えられていない場合は"/"
                pass
        else:  # 一般向け
            local_path = sum_proc["csv_input"][2]
            sum_proc["wbclient"]["local_path"] = local_path
            replace_flag = sum_proc["csv_input"][3]
            sum_proc["wbclient"]["replace_flag"] = replace_flag
            storage_provider = None
            if len(sum_proc["csv_input"]) > 4:
                storage_provider = sum_proc["csv_input"][4]
                sum_proc["wbclient"]["storage_provider"] = storage_provider
            else:  # 「storage_provider」がCSVで与えられていない場合は"osfstorage"
                sum_proc["wbclient"]["storage_provider"] = "osfstorage"
            remote_dir = None
            if len(sum_proc["csv_input"]) > 5:
                remote_dir = sum_proc["csv_input"][5]
                sum_proc["wbclient"]["remote_dir"] = remote_dir
            else:  # 「remote_dir」がCSVで与えられていない場合は"/"
                pass
        # check
        #print(f'[check:invoke_wbclient] local_path   = {local_path}')
        #print(f'[check:invoke_wbclient] replace_flag = {replace_flag}')
        #print(f'[check:invoke_wbclient] storage_provider = {storage_provider}')
        #print(f'[check:invoke_wbclient] remote_dir = {remote_dir}')
        
        # 環境変数(WBCLIENT_STORAGE_PROVIDER)の更新 -> TODO: 関数の引数で渡す形にしたい
        if storage_provider:  # CSVで指定されている場合は上書き
            os.environ["WBCLIENT_STORAGE_PROVIDER"] = storage_provider
        else:  # CSVで指定されていない場合はそのまま
            pass
        
        # r2dms_dmsutilのステータスが"SKIPPED"で、アップロード先がRIKEN S3の場合はスキップ
        if GRDM_RIKEN:  # 理研向けのみ
            if sum_proc["r2dms_dmsutil"]["status"] == "SKIPPED":
                #if sum_subproc["storage_provider"] == "s3compatriken":
                if sum_proc["wbclient"]["storage_provider"] == "s3compatriken":
                    mystat = "SKIPPED"
                    sum_proc["wbclient"]["message"] = "S3 bucket not avaialbe for data upload."
                    fin_summary_subproc(sum_proc["wbclient"], mystat)
                    return
                else:
                    pass
            else:
                pass
        else:
            pass
        
        # アップロードデータがNoneの場合はスキップ
        if not local_path:
            mystat = "SKIPPED"
            sum_proc["wbclient"]["message"] = "Local data file/directory path not given."
            fin_summary_subproc(sum_proc["wbclient"], mystat)
            return
        else:
            pass
        
        # 上書きフラグが不正な値(0,1以外)の場合はエラー
        flg_valid = True
        try:
            rf_int = int(replace_flag)
            if rf_int < 0 or 1 < rf_int:
                flg_valid = False
            else:
                pass
        except:
            flg_valid = False
            ## check
            #print(f'[check:invoke_wbclient] non-integer replace_flag = {replace_flag}')
        if not flg_valid:
            mystat = "NG"
            sum_proc["wbclient"]["message"] = f"Invalid `replace_flag` value (= {replace_flag}) given."
            fin_summary_subproc(sum_proc["wbclient"], mystat)
            return
        else:
            pass           
        ## check
        #print(f'[check:invoke_wbclient] rf_int       = {rf_int}')
        #print(f'[check:invoke_wbclient] type(rf_int) = {type(rf_int)}')
        
        # アップロードデータの存在チェック(wbclientの前にここでやっておく)
        if not os.path.exists(local_path):
            mystat = "NG"
            sum_proc["wbclient"]["message"] = f"Local data path (= {local_path}) not found."
            fin_summary_subproc(sum_proc["wbclient"], mystat)
            return
        else:
            pass
        
        # アップロード先のプロジェクト取得
        if sum_proc["create_grdmprj"]["status"] == "SKIPPED":  # 既存プロジェクトにデータをアップロード
            project_id = sum_proc["create_grdmprj"]["parent_guid"]
        else:  # 新しく作成されたプロジェクトにデータをアップロード
            project_id = sum_proc["create_grdmprj"]["guid_created"]
        sum_proc["wbclient"]["target_project_id"] = project_id  
        ## check
        #print(f'[check:invoke_wbclient] project_id = {project_id}')
        
        # アップロード先のディレクトリ取得
        remote_path = sum_proc["wbclient"]["remote_dir"]
        if not remote_path.startswith('/'):  # 先頭が「/」で始まっていない場合は「/」を挿入
            remote_path = '/' + remote_path
        else:
            pass
        if not remote_path.endswith('/'):  # 末尾が「/」で終わっていない場合は「/」を追加
            remote_path = remote_path + '/'
        else:
            pass
        sum_proc["wbclient"]["remote_dir"] = remote_path
        
        # wbclientの呼び出し(※alt版はwbclientからの返値がある)
        #import wbclient  # wbclient用の環境変数を更新しているので、ここでimportする  # poetry版
        from . import wbclient  # wbclient用の環境変数を更新しているので、ここでimportする  # exe化の際に「.」が必要
        exception = wbclient.exceptions
        command = wbclient.upload.UploadCommand()
        try:
            result = command.upload(project_id, local_path, rf_int, remote_path)
            sum_subproc["upload_result"] = result
            mystat = "OK"
            mymsg  = "Completed."
        except exception.BadRequestException as e:
            mystat = "NG"
            mymsg  = f"Error: {e}"
            ## check
            #print('[check:invoke_wbclient] except = "BadRequestException"')
        except exception.IllegalArgumentException as e:
            mystat = "NG"
            mymsg  = f"Error: {e}"
            ## check
            #print('[check:invoke_wbclient] except = "IllegalArgumentException"')
        except exception.WBClientException as e:  # このexceptだけでもよい
            mystat = "NG"
            mymsg  = f"Error: {e}"
            ## check
            #print('[check:invoke_wbclient] except = "WBClientException"')
        
        # 親プロセスのサマリを更新して呼び出し元に戻す
        fin_summary_subproc(sum_proc["wbclient"], mystat, mymsg)
        return

    # 全体制御(メイン関数)
    def controller(self, summary):  # summaryに逐次情報を書き込むので、関数の引数で渡すように変更
        
        ## debug
        #print(1/0)
        
        # 準備
        #n_proc = {}  # 処理数に関する辞書型データを準備
        #n_proc["id"] = 0  # 処理番号(0スタート) = 総処理数
        #n_proc["ok"] = 0  # 成功処理数 
        #n_proc["ng"] = 0  # 失敗処理数
        #n_proc["sk"] = 0  # スキップ処理数
        #cnum = 0  # 現在の処理番号(0スタート)
        #summary["processes"] = list()  # 処理ごとのサマリを配列で格納する
        
        # サマリデータの初期化
        #summary = R2dmsUploader().ini_summary()  # -> __main__.pyへ移動
        
        # 実行時引数の取得
        myargs = R2dmsUploader().get_execargs()
        inputfile  = myargs.input
        outputfile = myargs.output
        # check
        #print(f'[check:controller] inputfile  = {inputfile}')
        #print(f'[check:controller] outputfile = {outputfile}')
        summary["input_filepath"] = inputfile
        
        # 環境変数のチェック (トークンのチェックから拡張・移動)
        rc_env = R2dmsUploader().exam_environ()
        if rc_env !=0:
            return rc_env
        else:
            pass
        
        # 入出力ファイルのチェック
        rc_exam = R2dmsUploader().exam_inout(inputfile, outputfile)
        if rc_exam != 0:
            return rc_exam
        else:
            pass
        summary["output_filepath"] = outputfile
        
        # CSVデータの読み取り
        rc_csv, csvinputs = R2dmsUploader().read_inputs(inputfile)
        ## check
        #print(f'[check:controller] rc_csv = {rc_csv}')
        if rc_csv != 0:
            return rc_csv
        else:
            if csvinputs is not None:
                n_csvline = len(csvinputs)  # CSVファイルの有効行数
            else:
                return -17  # read_inputs関数内の最後のエラーコードが「-16」なので
        summary["num_csvline_valid"] = n_csvline
        ## check
        #print(f'[check:controller] n_csvline = {n_csvline}')
        
        ## トークンのチェック (-> 環境変数のチェックへ)
        #rc_tkn = R2dmsUploader().exam_token()
        #if rc_tkn !=0:
        #    return rc_tkn
        #else:
        #    pass
        
        ## debug
        #print(1/0)
        
        # CSVファイルの有効行でループ処理
        cnum = 0  # 現在の処理番号(0スタート)
        summary["processes"] = list()  # 処理ごとのサマリを配列で格納する準備
        for ins in csvinputs:
            
            n_proc = cnum + 1  # サマリ用の処理ID(1スタート)
            
            ## debug
            #print(1/0)
        
            # 今回の処理のサマリデータの初期化
            sum_proc_tmp = R2dmsUploader().ini_summary_proc(n_proc, ins)
            
            # 処理中の入力データを標準出力に表示(サマリをファイルに出力する場合のみ)
            # 処理結果を追記するので改行しない
            if outputfile:
                print(f'Now Processing : {sum_proc_tmp["proc_id"]} | CSV_input = {sum_proc_tmp["csv_input"]}', end='', flush=True)
            else:
                pass
            
            # 全体サマリに今回の処理のサマリを追加
            summary["processes"].append(sum_proc_tmp)
            
            # 既存プロジェクトIDと新規作成プロジェクトタイトルの両方がNoneの場合は、処理をスキップ
            parent_id = ins[0]
            prj_title = ins[1]
            ## check
            #print(f'[check:controller] parent_id = {parent_id}')
            #print(f'[check:controller] prj_title = {prj_title}')
            if not parent_id and not prj_title:
                mysta = "SKIPPED"
                mymsg = "Neither `parent_id` nor `prj_title` given; This process skipped."
                R2dmsUploader().fin_summary_proc(summary["processes"][cnum], mysta, mymsg)
                summary["num_proc_total"] += 1
                summary["num_proc_skipped"] += 1
                # 処理結果を標準出力に表示(サマリをファイルに出力する場合のみ)
                if outputfile:
                    print(f' -> {mysta}', flush=True)
                else:
                    pass
                cnum += 1  # 次の処理番号へ(+1)
                continue
            else:
                pass
            
            ## debug
            #print(1/0)
            
            ## check
            #time.sleep(10)
            #print('[check:controller] invoke_project : "create_grdmprj"')
            
            # 外部ツール呼び出し(1)：GRDMプロジェクトの作成
            R2dmsUploader().invoke_create_grdmprj(summary["processes"][cnum])
            if summary["processes"][cnum]["create_grdmprj"]["status"] == "NG":
                mysta = "NG"
                if GRDM_RIKEN:  # 理研向け
                    mymsg = "Error at `create_grdmprj`; `r2dms_dmsutil` and `wbclient` skipped."
                else:  # 一般向け
                    mymsg = "Error at `create_grdmprj`; `wbclient` skipped."
                R2dmsUploader().fin_summary_proc(summary["processes"][cnum], mysta, mymsg)
                summary["num_proc_total"] += 1
                summary["num_proc_failed"] += 1
                # 処理結果を標準出力に表示(サマリをファイルに出力する場合のみ)
                if outputfile:
                    print(f' -> {mysta}', flush=True)
                else:
                    pass
                cnum += 1  # 次の処理番号へ(+1)
                continue
            else:
                pass
            
            ## debug
            #print(1/0)
            
            ## check
            #time.sleep(10)
            #print('[check:controller] invoke_project : "r2dms_dmsutil"')
            
            # 外部ツール呼び出し(2)：S3バケットの作成およびGRDMプロジェクトへのアドオン
            if GRDM_RIKEN:  # 理研向けのみ
                ## check
                #print(f'[check:controller] With RIKEN Option')
                R2dmsUploader().invoke_r2dms_dmsutil(summary["processes"][cnum])
                if summary["processes"][cnum]["r2dms_dmsutil"]["status"] == "NG":  
                    mysta = "NG"
                    mymsg = "Error at `r2dms_dmsutil`; `wbclient` skipped."
                    R2dmsUploader().fin_summary_proc(summary["processes"][cnum], mysta, mymsg)
                    summary["num_proc_total"] += 1
                    summary["num_proc_failed"] += 1
                    # 処理結果を標準出力に表示(サマリをファイルに出力する場合のみ)
                    if outputfile:
                        print(f' -> {mysta}', flush=True)
                    else:
                        pass
                    cnum += 1  # 次の処理番号へ(+1)
                    continue
                else:
                    pass
            else:
                ## check
                #print(f'[check:controller] W/O RIKEN Option')
                pass
            
            ## debug
            #print(1/0)
            
            ## check
            #time.sleep(10)
            #print('[check:controller] invoke_project : "wbclient"')
            
            # 外部ツール呼び出し(3)：データアップロード
            R2dmsUploader().invoke_wbclient(summary["processes"][cnum])
            if summary["processes"][cnum]["wbclient"]["status"] == "NG":
                mysta = "NG"
                mymsg = "Error at `wbclient`."
                R2dmsUploader().fin_summary_proc(summary["processes"][cnum], mysta, mymsg)
                summary["num_proc_total"] += 1
                summary["num_proc_failed"] += 1
                # 処理結果を標準出力に表示(サマリをファイルに出力する場合のみ)
                if outputfile:
                    print(f' -> {mysta}', flush=True)
                else:
                    pass
                cnum += 1  # 次の処理番号へ(+1)
                continue
            else:
                pass
            
            # 今回の処理は正常終了としてサマリデータを更新
            mysta = 'OK'
            mymsg = 'Completed.'
            R2dmsUploader().fin_summary_proc(summary["processes"][cnum], mysta, mymsg)
            summary["num_proc_total"] += 1
            summary["num_proc_succeeded"] += 1
            # 処理結果を標準出力に表示(サマリをファイルに出力する場合のみ)
            if outputfile:
                print(f' -> {mysta}', flush=True)
            else:
                pass
            
            cnum += 1  # 次の処理番号へ(+1)
            
        # forループの終了
        
        # 全ての入力に対する処理が完了したことを標準出力に表示(サマリをファイルに出力する場合のみ)
        if outputfile:
            print('All processes completed.', flush=True)
            if summary["num_proc_failed"] != 0 or summary["num_proc_skipped"] != 0:
                print('WARNING : Check summary of NG/SKIPPED process(es)!', flush=True)
            else:
                pass
        else:
            pass
            
        # 呼び出し元にリターンコード(0=正常)を返す
        return 0
        


        
        
